/*****************************************************************************
*  LPC54xx SCT PWM reload program
*
*  Use the match reload registers to change the duty cycle of two PWM
*  signals and maintain their dead-time intervals using the NORELOAD_L bit
*  in the SCT Configuration register
*****************************************************************************/

#include "board.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
#define DC1        (130)                                   				// duty cycle 1
#define DC2        (135)                                   				// duty cycle 2
#define hperiod    (180)

/*****************************************************************************
 * Private functions
 ****************************************************************************/
extern void SCT_Init(void);
/*****************************************************************************
 * Public functions
 ****************************************************************************/

void SCT_Init(void)
{

	Chip_SCT_Init(LPC_SCT);			                  		 											// enable the SCT clock

	Chip_SCT_Config(LPC_SCT, SCT_CONFIG_16BIT_COUNTER  |												// split timers, auto limit
                            SCT_CONFIG_AUTOLIMIT_L    );

	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_BIDIR_L(1));     												// configure SCT as BIDIR

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, hperiod);												// match on (half) PWM period
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, hperiod);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_1, DC1);													// match on duty cycle 1
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, DC1);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_2, DC2);													// match on duty cycle 2
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, DC2);

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);	 									// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1  |
                                                                        SCT_COMBMODE_MATCH  ));			// match 1 (DC1) only condition

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);	 									// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1,  (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH2 |
                                                                          SCT_COMBMODE_MATCH  )); 		// match 2 (DC) only condition

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_0);												// event 0 set OUT4 (blue LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_0);												//  event 0 clears OUT4 (blue LED)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_5, SCT_EVT_1);												// event 1 sets OUT5 (green LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_5, SCT_EVT_1); 											// event 1 clear OUT5 (green LED)

	Chip_SCT_SetConflictResolution(LPC_SCT, 4,SCT_RES_TOGGLE_OUTPUT);  									// toggle OUT4 on conflict

	Chip_SCT_SetConflictResolution(LPC_SCT, 5,SCT_RES_TOGGLE_OUTPUT);									// toggle OUT5 on conflict

	Chip_SCT_Output(LPC_SCT, (1 << 4));                   												// default set OUT4 and and clear OUT5


    Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   													// unhalt it by clearing bit 2 of the CTRL register
}


void SysTick_Handler(void)
{

	Chip_SCT_SetControl(LPC_SCT, SCT_CONFIG_NORELOADL_U);												// stop reload process for L counter

    if (Chip_GPIO_GetPinState(LPC_GPIO, 0, 24))              											// P0_24 high?
    {
        if (LPC_SCT->MATCHREL[2].L < hperiod-1)        													// check if DC2 < Period of PWM
        {
            LPC_SCT->MATCHREL[1].L ++;
            LPC_SCT->MATCHREL[2].L ++;
        }
    }
    else if (LPC_SCT->MATCHREL[1].L > 1)              													// check if DC1 > 1
    {
        LPC_SCT->MATCHREL[1].L --;
        LPC_SCT->MATCHREL[2].L --;
    }
    LPC_SCT->CONFIG &= ~(1 << 7);                     													// enable reload process for L counter
}

int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCON_CLOCK_T) ( SYSCON_CLOCK_GPIO1 	| 						// enable GPIO port 1 clock
                                                        	SYSCON_CLOCK_GPIO0 	|						// enable GPIO port 2 clock
                                                        	SYSCON_CLOCK_IOCON	));		    			// enable IOCON clock

	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 2, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT5 = P1.2  (green   LED)
	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 1, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT4 = P1.1  (blue   LED)

	SysTick_Config(SystemCoreClock/100);                												// SysTick interrupt @ 120 Hz = 100 msec
	SCT_Init();                                          												// Initialize SCT0



    while (1)                                              			// loop forever
    {
    	__WFI();
    }
}
